import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { productSchema } from '@/lib/validations';
import { decrementUsage } from '@/lib/usage';

export async function PATCH(
  req: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session?.user) {
      return NextResponse.json({ error: 'Neautentificat' }, { status: 401 });
    }

    const restaurant = await prisma.restaurant.findFirst({
      where: { ownerId: session.user.id },
    });

    if (!restaurant) {
      return NextResponse.json({ error: 'Restaurant negăsit' }, { status: 404 });
    }

    const product = await prisma.product.findUnique({
      where: { id: params.id },
    });

    if (!product || product.restaurantId !== restaurant.id) {
      return NextResponse.json({ error: 'Produs negăsit' }, { status: 404 });
    }

    const body = await req.json();
    const data = productSchema.partial().parse(body);

    const updated = await prisma.product.update({
      where: { id: params.id },
      data,
    });

    await prisma.auditLog.create({
      data: {
        userId: session.user.id,
        action: 'UPDATE',
        entity: 'product',
        entityId: updated.id,
        metadata: { name: updated.nameRo },
      },
    });

    return NextResponse.json(updated);
  } catch (error) {
    console.error('Update product error:', error);
    return NextResponse.json({ error: 'Eroare server' }, { status: 500 });
  }
}

export async function DELETE(
  req: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session?.user) {
      return NextResponse.json({ error: 'Neautentificat' }, { status: 401 });
    }

    const restaurant = await prisma.restaurant.findFirst({
      where: { ownerId: session.user.id },
      include: { subscription: true },
    });

    if (!restaurant) {
      return NextResponse.json({ error: 'Restaurant negăsit' }, { status: 404 });
    }

    const product = await prisma.product.findUnique({
      where: { id: params.id },
    });

    if (!product || product.restaurantId !== restaurant.id) {
      return NextResponse.json({ error: 'Produs negăsit' }, { status: 404 });
    }

    await prisma.product.delete({
      where: { id: params.id },
    });

    if (restaurant.subscription) {
      await decrementUsage(restaurant.subscription.id, 'products', 1);
    }

    await prisma.auditLog.create({
      data: {
        userId: session.user.id,
        action: 'DELETE',
        entity: 'product',
        entityId: params.id,
        metadata: { name: product.nameRo },
      },
    });

    return NextResponse.json({ message: 'Produs șters' });
  } catch (error) {
    console.error('Delete product error:', error);
    return NextResponse.json({ error: 'Eroare server' }, { status: 500 });
  }
}